# Module UI
  
#' @title   mod_ciclo_completo_ui and mod_ciclo_completo_server
#' @description  A shiny Module.
#'
#' @param id shiny id
#' @param input internal
#' @param output internal
#' @param session internal
#'
#' @rdname mod_ciclo_completo
#'
#' @keywords internal
#' @export 
#' @importFrom shiny NS tagList 
mod_ciclo_completo_ui <- function(id){
  ns <- NS(id)
  ui <- uiOutput(ns('ciclo_completo'))
}
    
# Module Server
    
#' @rdname mod_ciclo_completo
#' @export
#' @keywords internal
    
mod_ciclo_completo_server <- function(input, output, session, banco){
  ns <- session$ns
  
  to_export <- reactiveValues()

  ## Busca valores de referencia
  
  get_desembolso <- reactive({
    
    indicador <- 'DESEMBOLSO CAB/MÊS'
    
    ref_var <- banco$regua %>% filter(INDICADOR == indicador) %>% arrange(safra) %>% tail(1)
    
    banco$ind_top %>% filter(INDICADOR == indicador) -> config_var
    
    domain <- ref_var %>% select(min, q25, q50, q75, max) %>% unlist %>% as.vector
    referencia <- ref_var$ref
    bt_value <- 1
    valor_fazenda <- banco$fazenda %>% filter(INDICADOR == indicador) %>% arrange(safra) %>% tail(1) %>% pull(VALOR)
    
    if (config_var$TOP == 'MENOR') {
      
      bt_value <- 0
      domain <- rev(domain)
      
    }
    
    list(
      label = "Desembolso",
      unidade = "R$/cab/mês",
      value = valor_fazenda, 
      domain = domain, 
      entity = list(id = "fazenda", label = "Fazenda", value = valor_fazenda),
      reference = list(id = 'reference', label = "Referência", value = referencia),
      bt = bt_value, 
      casas = config_var %>% pull(casas)
    ) 
      
    
    
  })
  
  ### ÁREA E LOTAÇÃO

  rebanho <- reactive({input$area_produtiva * input$lotacao_cabeca}) 
  
  ### PARÂMETROS ZOOTÉCNICOS
  # Variáveis reativas declaradas fora de funções, estão assim porque são utilizadas em
  # mais de uma função
  
  matrizes <- reactive({
    rebanho() * input$participacao_total / 100
  })
  vacas_prenhes <- reactive({
    matrizes() * input$fertilidade / 100
  })
  bezerros_nascidos <- reactive({vacas_prenhes() * (1 - input$perda_pre_parto/100)})
  vaca_descarte <- reactive({matrizes() - bezerros_nascidos()})
  bezerros_desmamados <- reactive({bezerros_nascidos() * (1 - input$mortalidade_bezerros/100)})
  desmamados_machos <- reactive({bezerros_desmamados() / 2})
  bezerras_desmamadas <- reactive({bezerros_desmamados() - desmamados_machos()})
  venda_bezerras_femeas <- reactive({bezerras_desmamadas() * input$venda_bezerras_femeas / 100})
  venda_bezerros_machos <- reactive({desmamados_machos() * input$venda_bezerros_machos / 100})
  novilhas_disponiveis <- reactive({bezerras_desmamadas() - venda_bezerras_femeas() - vaca_descarte()})
  venda_bois <- reactive({desmamados_machos() - venda_bezerros_machos()})
  
  calcZoot <- reactive({
    tibble(
      descricao = c("Rebanho","Matrizes", "Vacas prenhes", "Bezerros nascidos", "Total de bezerros desmamados",
                    "Bezerros desmamados machos", "Venda de bezerros", "Bezerras desmamadas",
                    "Venda de bezerras", "Novilhas disponíveis", "Venda de bois"),
      unidade = c("cabeça"),
      valor = c(
        rebanho(),
        matrizes(), 
        vacas_prenhes(),
        bezerros_nascidos(), 
        bezerros_desmamados(),
        desmamados_machos(),
        venda_bezerros_machos(),
        bezerras_desmamadas(),
        venda_bezerras_femeas(),
        novilhas_disponiveis(),
        venda_bois()
      )
    )
  })
  
  output$zooT <- function(){
    
    calcZoot() -> to_export[['zoot']]

    knitr::kable(calcZoot(), escape = F, format("html"), digits = 0,
                 col.names = c("Descrição", "Unidade", "Quantidade")) %>%
                row_spec(row = 0, background = "#c4c4c4") %>%
      kable_styling(bootstrap_options = c("striped", "bordered","condensed")) 
  }
  
  
  ### ESTIMATIVAS
    
  # variaveis reativas
  idade_nao_entouradas <- reactive({
    input$idade_nao_entouradas
  })
  femeas_13_24 <- reactive({
    f1 <- bezerras_desmamadas() - venda_bezerras_femeas() - novilhas_disponiveis()
    f2 <- (idade_nao_entouradas() - 12) / 12 * novilhas_disponiveis()
    (f1 + f2)
  })
  machos_8_12 <- reactive({
    (desmamados_machos() - venda_bezerros_machos()) * (12 - input$idade_desmame) / 12
  })
  machos_13_24 <- reactive({
    #ifelse(venda_bois() > 24, machos_8_12(), input$idade_abate/24*machos_8_12())
    ifelse(input$idade_abate > 24, venda_bois(), venda_bois() * (input$idade_abate - 12)/12)
  })
  machos_mais_24 <- reactive({ 
    ifelse(input$idade_abate <= 24, 0, (input$idade_abate - 24) * (venda_bois()/12))
  })
  touros <- reactive({
    matrizes() * 0.04 
  })
  
  estimativas <- reactiveValues()
  
  estimativaRebanhos <- function(){
    
    cabecas <- c(matrizes(), 
                 bezerros_desmamados()*input$idade_desmame/12, # bezerros mamando
                 # correcao: subtracao venda_bezerras_femeas
                 ((bezerras_desmamadas() - venda_bezerras_femeas()) * (12 - input$idade_desmame) / 12), # femeas 8 a 12
                 femeas_13_24(), # 
                 machos_8_12(),
                 machos_13_24(),
                 machos_mais_24(),
                 touros()
    )
    
    pesos <- c(
      input$peso_matrizes,
      input$peso_bezerros_mamando,
      input$peso_f_12,
      input$peso_f_24,
      input$peso_m_12,
      input$peso_m_24,
      input$peso_m_mais,
      input$peso_touros
    )
    
    sum_prod_peso_cab <- sum(pesos*cabecas)
    
    dados <- tibble(
      categoria = c("Matrizes", "Bezerros mamando", "Fêmeas de 8 a 12 meses",
                    "Fêmeas de 13 a 24 meses", "Machos de 8 a 12 meses", "Machos de 13 a 24 meses",
                    "Machos com mais de 24 meses", "Touros"),
      cabecas = cabecas 
    ) 
    
    estimativas$total_cab <- sum(dados$cabecas)
    dados %>% 
      add_row(categoria = "Total (cabeças)", cabecas = estimativas$total_cab) %>% 
      add_row(categoria = "Total (peso)", cabecas = sum(pesos*cabecas)) %>%
      mutate(cabecas = format(round(cabecas, digits = 0), digits = 0, nsmall = 0)) %>% 
      add_row(categoria = "LOTAÇÃO/HA (cabeças)", cabecas = format(round(estimativas$total_cab/input$area_produtiva, digits = 2),digits = 2, nsmall = 2, decimal.mark = ',')) %>%
      add_row(categoria = "LOTAÇÃO/HA (UA)", cabecas = format(round(sum_prod_peso_cab/input$area_produtiva, digits = 2), digits = 2, nsmall = 2, decimal.mark = ','))
  }
  
  output$estimativas <- function(){
    dados <- estimativaRebanhos()
    
    dados -> to_export[['estimativas']]
    
    dados %>% 
      knitr::kable( col.names = c("Categoria", "Cabeças"), escape = F, digits = 2, format = "html", align = 'lr') %>%
      kable_styling(c("bordered","condensed", "striped", "hover")) %>%
      row_spec(row = 9:12, bold = T, background = "#479860", color = "white", font_size = "14px") %>%
      row_spec(row = 0, background = "#c4c4c4")
  }
  
  ### SEÇÃO DE VENDAS

  dados_vendas <- reactiveValues()
  
  # tabela de vendas
  
  calcVendas <- function(){
    # browser()
    arroba_descarte <- reactive({ input$venda_vc_peso * input$rend_carc_vc/100/15 })
    arroba_venda_bois <- reactive({ input$venda_peso_bois * input$rend_carc_bois/100 /15 })
    arroba_venda_novilha <- reactive({ input$venda_peso_novilhas * input$rend_carc_novilhas/100 / 15})
    arroba_venda_touros <- reactive({ input$venda_peso_touros * input$rend_carc_touros/100 / 15})
    
    dados_vendas$touros <- touros() * 0.2
  
    
    data.frame(
      row.names = c("Vacas descarte", "Desmame macho (Kg)", "Desmame fêmeas (Kg)",
                    "Venda de bois", "Novilhas descarte", "Touros descarte"),
      quantidade = c(matrizes() - bezerros_nascidos(), venda_bezerros_machos(), venda_bezerras_femeas(),
                     venda_bois(), novilhas_disponiveis(), dados_vendas$touros),
      arroba = c(arroba_descarte(), NA, NA, arroba_venda_bois(), arroba_venda_novilha(), arroba_venda_touros()),
      valcab = c(
        arroba_descarte() * input$venda_descarte_valor, 
        input$venda_macho_peso * input$venda_macho_valor,
        input$venda_femea_peso * input$venda_femea_valor, 
        arroba_venda_bois() * input$venda_bois_valor,
        arroba_venda_novilha() * input$venda_novilhas_valor, 
        arroba_venda_touros() * input$venda_touros_valor
        )
    ) %>% rownames_to_column('cat') %>% mutate(
      quantidade = round(quantidade, digits=0), totalcat = valcab*quantidade
    ) -> df
    
  
  }

  output$calcVendas <- reactive({
    
    df <- calcVendas() 
    dados_vendas$receita <- sum(df$totalcat)
    df %>% 
      bind_rows(
        df %>% 
          summarise(cat="Total", quantidade=sum(quantidade), totalcat=sum(totalcat))
      ) %>% 
      mutate(
        arroba = ifelse(is.na(arroba), "", format(round(arroba,2), nsmall = 2L, decimal.mark = ",", big.mark = ".")), 
        valcab = ifelse(is.na(valcab), "", format(round(valcab,2), nsmall=2L, decimal.mark = ',', big.mark = '.')),
        totalcat = format(round(totalcat, digits=2L), nsmall = 2L, decimal.mark = ',', big.mark = '.')
      ) %>% 
      column_to_rownames('cat') -> vendas_calc
    
    vendas_calc -> to_export[['vendas_calc']]
    
    vendas_calc %>% 
      knitr::kable(align = 'rrrr',
                   col.names = c("Quantidade", "Peso (@)", "Valor/cab", "Valor total")
                   ) %>% 
      kable_styling(c('bordered', 'condensed','striped', 'hover')) %>% 
      #add_header_above(c("TABELA DE VENDAS"=5), bold = T) %>%
      row_spec(0, background = "#c4c4c4", bold=T) %>%
      row_spec(7, bold = T, background = "#479860", color = "white", font_size = "14px")
  })
    
  # TABELA RESULTADO
  output$resultados<- reactive({
    
    desemb <- estimativas$total_cab * input$desembolso_cab_mes * 12
    tou <- dados_vendas$touros * 7500
    data.frame(c("Receita", "Desembolso/Projeto/Ano", "Reposição de touros", "Total",
                 "Resultado da operação anual","Resultado da operação anual/HA"),
               c(
                 dados_vendas$receita, 
                 desemb,
                 tou, desemb+tou, dados_vendas$receita-desemb-tou,
                 (dados_vendas$receita-desemb-tou) / input$area_produtiva
               ) %>% format(digits=2, nsmall = 2L, big.mark = '.', decimal.mark = ',', zero.print = "R$")
    ) -> resultados
    
    resultados -> to_export[['resultados']]
    
    resultados %>% 
      knitr::kable(col.names = c("Resultado", "R$"), align = "lr", digits=2) %>%
      row_spec(0, background = "#c4c4c4", bold=T) %>% 
      kable_styling(bootstrap_options = c("condensed", "bordered", "striped", "hover"))
    
  })
  
  output$ciclo_completo <- renderUI({
    
    last_safra <- banco$prospectiva$ciclo_completo$safra %>% unique %>% na.omit %>% sort %>% last
    last_safra_f <- banco$fazenda$safra %>% unique %>% na.omit %>% sort %>% last
    
    variaveis <- c(
      'ÁREA TOTAL FAZENDA',
      'LOTAÇÃO GLOBAL UA/HA',
      'LOTAÇÃO GLOBAL CAB/HA',
      'MORTALIDADE BEZERROS',
      'DESEMBOLSO CAB/MÊS',
      'RENDIMENTO DE CARCAÇA MACHOS',
      'RENDIMENTO DE CARCAÇA NOVILHAS',
      'RENDIMENTO DE CARCAÇA DE MATRIZES DESCARTE',
      'FERTILIDADE DE GERAL (SAFRA FUTURA)',
      'PERDA PRÉ-PARTO (SAFRA ATUAL)',
      # 'PESO MÉDIO DOS BEZERROS DESMAMADOS (KG)',
      'VALOR @ MACHOS',
      # 'VALOR MÉDIO DE VENDA',
      'PESO AO DESMAME FÊMEAS NELORE (kg)',
      'PESO AO DESMAME MACHOS NELORE (kg)',
      'VALOR @ NOVILHAS'
    )
    
    complete_farm <- function(fazenda, ref) {
      setdiff(names(ref), names(fazenda)) -> faltantes
      
      if(length(faltantes) > 0) {
      
        for(i in 1:length(faltantes)) {
          
          #i<-1
          faltantes_i <- faltantes[i]
          ref[[faltantes_i]]$media -> fazenda[[faltantes_i]]
          
        }
        
      }
      return(fazenda)
    }
    
    round_inttegra <- function(x, unidade, casas) {
      
      if (!is.na(unidade)) {
        if (unidade == '%') {
          x = x*100
        }  
      }
      
      if (is.na(casas)) {
        x
      } else {
        round(x, casas)
      }
    }

    ref_farm <- banco$fazenda %>% 
      left_join(
        banco$ind_top %>% 
          select(FRENTE, INDICADOR, casas, unidade),
        by = c("FRENTE", "INDICADOR")
      ) %>% 
      filter(safra == last_safra_f) %>% 
      filter(INDICADOR %in% variaveis) %>% 
      select(INDICADOR, VALOR, casas, unidade) %>% 
      rowwise %>%  
      mutate(VALOR = pmap_dbl(.l = list(VALOR, unidade, casas), .f = round_inttegra)) %>% 
      ungroup %>% 
      select(-casas, -unidade) %>% 
      spread(key = INDICADOR, value = VALOR) %>% 
      as.list
    
    ref_cre <- banco$prospectiva$ciclo_completo %>% 
      left_join(
        banco$ind_top %>% 
          select(FRENTE, INDICADOR, casas),
        by = c("FRENTE", "INDICADOR")
      ) %>% 
      filter(safra == last_safra) %>% 
      filter(INDICADOR %in% variaveis) %>% 
      select(INDICADOR, media, top30 = ref, topind = q70, unidade, casas) %>% 
      rowwise %>%  
      mutate(media = pmap_dbl(.l = list(media, unidade, casas), .f = round_inttegra)) %>% 
      mutate(top30 = pmap_dbl(.l = list(top30, unidade, casas), .f = round_inttegra)) %>% 
      mutate(topind = pmap_dbl(.l = list(topind, unidade, casas), .f = round_inttegra)) %>% 
      ungroup %>% 
      select(-casas, -unidade) %>% 
      split(.$INDICADOR) %>% 
      lapply(., gather, key = 'chave', value = 'valor', -INDICADOR) %>% 
      lapply(select, - INDICADOR) %>% 
      lapply(spread, key = chave, value = valor) %>% 
      lapply(as.list)
    
    ref_farm <- complete_farm(ref_farm, ref_cre)
    
    ctrlAreaLotacao <- lapiCollapsablePanel(
      ns("caminhos"), "Parâmetros gerais",
      lapiNumericInput(inputId = ns('area_produtiva'), label = "Área Produtiva", value = ref_farm$`ÁREA TOTAL FAZENDA`),
      lapiSliderInput(inputId = ns('lotacao_area'), label="Lotação, UA/ha",  min=0, max=5, value=ref_farm$`LOTAÇÃO GLOBAL UA/HA`, step=0.01,
                      tooltip = write_ref_tooltip(
                        fazenda = ref_farm$`LOTAÇÃO GLOBAL UA/HA`, 
                        media = ref_cre$`LOTAÇÃO GLOBAL UA/HA`$media, 
                        top30 = ref_cre$`LOTAÇÃO GLOBAL UA/HA`$top30, 
                        topind = ref_cre$`LOTAÇÃO GLOBAL UA/HA`$topind
                      )
      ),
      lapiSliderInput(inputId = ns('lotacao_cabeca'), label="Lotação, cab/ha", min=0, max=5, value=2.7, step=0.01,
                      tooltip = write_ref_tooltip(
                        fazenda = ref_farm$`LOTAÇÃO GLOBAL CAB/HA`, 
                        media = ref_cre$`LOTAÇÃO GLOBAL CAB/HA`$media, 
                        top30 = ref_cre$`LOTAÇÃO GLOBAL CAB/HA`$top30, 
                        topind = ref_cre$`LOTAÇÃO GLOBAL CAB/HA`$topind
                      )
      ),
      lapiSliderInput(inputId = ns('participacao_total'), label="Participação matrizes, % do rebanho", min=0, max=100, value=ref_farm$`LOTAÇÃO GLOBAL CAB/HA`, step=0.01),
      lapiSliderInput(inputId = ns('fertilidade'), label="Fertilidade, %", min=0, max=100, value=ref_farm$`FERTILIDADE DE GERAL (SAFRA FUTURA)` , step=0.01,
                      tooltip = write_ref_tooltip(
                        fazenda = ref_farm$`FERTILIDADE DE GERAL (SAFRA FUTURA)` , 
                        media = ref_cre$`FERTILIDADE DE GERAL (SAFRA FUTURA)`$media  , 
                        top30 = ref_cre$`FERTILIDADE DE GERAL (SAFRA FUTURA)`$top30  , 
                        topind = ref_cre$`FERTILIDADE DE GERAL (SAFRA FUTURA)`$topind 
                      )
      ),
      lapiSliderInput(inputId = ns('perda_pre_parto'), label="Perda pré parto, %", min=0, max=100, value=ref_farm$`PERDA PRÉ-PARTO (SAFRA ATUAL)`, step=0.01,
                      tooltip = write_ref_tooltip(
                        fazenda = ref_farm$`PERDA PRÉ-PARTO (SAFRA ATUAL)`, 
                        media = ref_cre$`PERDA PRÉ-PARTO (SAFRA ATUAL)`$media  , 
                        top30 = ref_cre$`PERDA PRÉ-PARTO (SAFRA ATUAL)`$top30  , 
                        topind = ref_cre$`PERDA PRÉ-PARTO (SAFRA ATUAL)`$topind 
                      )
      ),
      lapiSliderInput(inputId = ns('mortalidade_bezerros'), label="Mortalidade Bezerros, %", min=0, max=100, value=ref_farm$`MORTALIDADE BEZERROS`  , step=0.01,
                      tooltip = write_ref_tooltip(
                        fazenda = ref_farm$`MORTALIDADE BEZERROS`  , 
                        media = ref_cre$`MORTALIDADE BEZERROS`$media  , 
                        top30 = ref_cre$`MORTALIDADE BEZERROS`$top30  , 
                        topind = ref_cre$`MORTALIDADE BEZERROS`$topind 
                      )
      ),
      lapiSliderInput(inputId = ns('idade_desmame'), label="Idade ao desmame, meses", min=0, max=12, value=7, step=1),
      lapiSliderInput(inputId = ns('venda_bezerros_machos'), label="Venda de bezerros machos, %", min=0, max=100, value=10, step=1),
      lapiSliderInput(inputId = ns('venda_bezerras_femeas'), label="Venda de bezerras fêmeas, %", min=0, max=100, value=20, step=1),
      lapiSliderInput(inputId = ns('idade_nao_entouradas'), label="Idade de venda de novilhas não entouradas, meses", min=0, max=120, value=20, step=1),
      lapiSliderInput(inputId = ns('idade_abate'), label="Idade ao abate, meses", min=0, max=120, value=26, step=1)
    )
    
    
    ctrlEstimativas <- lapiCollapsablePanel(
      id=ns("ctrlEstim"), "Parâmetros de peso do rebanho UA",
      lapiNumericInput(inputId = ns('peso_matrizes'), label="Matrizes", min=0, max=1, value=1, step=0.01),
      lapiNumericInput(inputId = ns('peso_bezerros_mamando'), label="Bezerros Mamando", min=0, max=1, value=0.25, step=0.01),
      lapiNumericInput(inputId = ns('peso_f_12'), label="Fêmeas 8 a 12 meses", min=0, max=1, value=0.45, step=0.01),
      lapiNumericInput(inputId = ns('peso_f_24'), label="Fêmeas 13 a 24 meses", min=0, max=1, value=0.6, step=0.01),
      lapiNumericInput(inputId = ns('peso_m_12'), label="Machos 8 a 12 meses", min=0, max=1, value=0.5, step=0.01),
      lapiNumericInput(inputId = ns('peso_m_24'), label="Machos 13 a 24 meses", min=0, max=1, value=0.9, step=0.01),
      lapiNumericInput(inputId = ns('peso_m_mais'), label="Machos +24 meses", min=0, max=2, value=1.2, step=0.01),
      lapiNumericInput(inputId = ns('peso_touros'), label="Touros", min=0, max=2, value=1.7, step=0.01)
    )
    
    ctrlVendasForm <- lapiCollapsablePanel(
      id = ns('vendasForm'), "Parâmetros de venda / abate", 
      tags$fieldset( class="lapi-fieldset",
                     tags$legend("Vacas Descarte"),
                     lapiNumericInput(inputId = ns('venda_vc_peso'), label="Peso (Kg)", value = 450),
                     lapiSliderInput(inputId = ns('rend_carc_vc'), label="Rendimento de carcaça", step=0.01, min=45, max=65, value= ref_farm$`RENDIMENTO DE CARCAÇA DE MATRIZES DESCARTE`  ,
                                     tooltip = write_ref_tooltip(
                                       fazenda = ref_farm$`RENDIMENTO DE CARCAÇA DE MATRIZES DESCARTE`  , 
                                       media = ref_cre$`RENDIMENTO DE CARCAÇA DE MATRIZES DESCARTE`$media  , 
                                       top30 = ref_cre$`RENDIMENTO DE CARCAÇA DE MATRIZES DESCARTE`$top30  , 
                                       topind = ref_cre$`RENDIMENTO DE CARCAÇA DE MATRIZES DESCARTE`$topind 
                                     )                
                     ),
                     lapiNumericInput(inputId = ns('venda_descarte_valor'), label="Valor (R$/@)", value = 160, step = 0.01)
      ),
      tags$fieldset(class="lapi-fieldset",
                    tags$legend("Desmame machos"),
                    lapiSliderInput(inputId = ns('venda_macho_peso'), label="Peso (kg)", step=1, min=100, max=350, value= ref_farm$`PESO AO DESMAME MACHOS NELORE (kg)`,
                                    tooltip = write_ref_tooltip(
                                      fazenda = ref_farm$`PESO AO DESMAME MACHOS NELORE (kg)`, 
                                      media = ref_cre$`PESO AO DESMAME MACHOS NELORE (kg)`$media, 
                                      top30 = ref_cre$`PESO AO DESMAME MACHOS NELORE (kg)`$top30, 
                                      topind = ref_cre$`PESO AO DESMAME MACHOS NELORE (kg)`$topind
                                    )                
                    ),
                    # lapiNumericInput(inputId = ns('venda_macho_peso'), label="Peso (Kg)", value = 221),
                    lapiNumericInput(inputId = ns('venda_macho_valor'), label="Valor (R$/Kg)", value = 10)
      ),
      tags$fieldset(class="lapi-fieldset",
                    tags$legend("Desmame fêmeas"),
                    lapiSliderInput(inputId = ns('venda_femeas_peso'), label="Peso (kg)", step=1, min=100, max=350, value= ref_farm$`PESO AO DESMAME FÊMEAS NELORE (kg)`,
                                    tooltip = write_ref_tooltip(
                                      fazenda = ref_farm$`PESO AO DESMAME FÊMEAS NELORE (kg)`, 
                                      media = ref_cre$`PESO AO DESMAME FÊMEAS NELORE (kg)`$media, 
                                      top30 = ref_cre$`PESO AO DESMAME FÊMEAS NELORE (kg)`$top30, 
                                      topind = ref_cre$`PESO AO DESMAME FÊMEAS NELORE (kg)`$topind
                                    )                
                    ),
                    lapiNumericInput(inputId = ns('venda_femea_peso'), label="Peso (Kg)", value = 204),
                    lapiNumericInput(inputId = ns('venda_femea_valor'), label="Valor (R$/Kg)", value = 10)
      ),
      tags$fieldset( class="lapi-fieldset",
                     tags$legend("Venda de bois"),
                     lapiNumericInput(inputId = ns('venda_peso_bois'), label="Peso (Kg)", value = 540),
                     lapiSliderInput(inputId = ns('rend_carc_bois'), label="Rendimento de carcaça", step=0.01, min=45, max=65, value= ref_farm$`RENDIMENTO DE CARCAÇA MACHOS`  ,
                                     tooltip = write_ref_tooltip(
                                       fazenda = ref_farm$`RENDIMENTO DE CARCAÇA MACHOS`  , 
                                       media = ref_cre$`RENDIMENTO DE CARCAÇA MACHOS`$media  , 
                                       top30 = ref_cre$`RENDIMENTO DE CARCAÇA MACHOS`$top30  , 
                                       topind = ref_cre$`RENDIMENTO DE CARCAÇA MACHOS`$topind 
                                     )                   
                     ),
                     lapiSliderInput(inputId = ns('venda_bois_valor'), label="Valor (R$/@)", step=0.01, min=100, max=350, value= ref_farm$`VALOR @ MACHOS`,
                                     tooltip = write_ref_tooltip(
                                       fazenda = ref_farm$`VALOR @ MACHOS`, 
                                       media = ref_cre$`VALOR @ MACHOS`$media, 
                                       top30 = ref_cre$`VALOR @ MACHOS`$top30, 
                                       topind = ref_cre$`VALOR @ MACHOS`$topind
                                     )                   
                     )
                     # lapiNumericInput(inputId = ns('venda_bois_valor'), label="Valor (R$/@)", value = 173.74, step = 0.01)
      ),
      tags$fieldset( class="lapi-fieldset",
                     tags$legend("Novilhas descarte"),
                     lapiNumericInput(inputId = ns('venda_peso_novilhas'), label="Peso (Kg)", value = 360),
                     lapiSliderInput(inputId = ns('rend_carc_novilhas'), label="Rendimento de carcaça", step=0.01, min=45, max=65, value= ref_farm$`RENDIMENTO DE CARCAÇA NOVILHAS`  ,
                                     tooltip = write_ref_tooltip(
                                       fazenda = ref_farm$`RENDIMENTO DE CARCAÇA NOVILHAS`  , 
                                       media = ref_cre$`RENDIMENTO DE CARCAÇA NOVILHAS`$media  , 
                                       top30 = ref_cre$`RENDIMENTO DE CARCAÇA NOVILHAS`$top30  , 
                                       topind = ref_cre$`RENDIMENTO DE CARCAÇA NOVILHAS`$topind 
                                     )                 
                     ),
                     lapiSliderInput(inputId = ns('venda_novilhas_valor'), label="Valor (R$/@)", step=0.01, min=100, max=350, value= ref_farm$`VALOR @ NOVILHAS`,
                                     tooltip = write_ref_tooltip(
                                       fazenda = ref_farm$`VALOR @ NOVILHAS`, 
                                       media = ref_cre$`VALOR @ NOVILHAS`$media, 
                                       top30 = ref_cre$`VALOR @ NOVILHAS`$top30, 
                                       topind = ref_cre$`VALOR @ NOVILHAS`$topind
                                     )                   
                     )
                     # lapiNumericInput(inputId = ns('venda_novilhas_valor'), label="Valor (R$/@)", value = 168.77, step = 0.01)
      ),
      tags$fieldset( class="lapi-fieldset",
                     tags$legend("Touros descarte"),
                     lapiNumericInput(inputId = ns('venda_peso_touros'), label="Peso (Kg)", value = 700),
                     lapiSliderInput(inputId = ns('rend_carc_touros'), label="Rendimento de carcaça", step=0.01, min=45, max=65, value=ref_farm$`RENDIMENTO DE CARCAÇA MACHOS`  , 
                                     tooltip = write_ref_tooltip(
                                       fazenda = ref_farm$`RENDIMENTO DE CARCAÇA MACHOS`  , 
                                       media = ref_cre$`RENDIMENTO DE CARCAÇA MACHOS`$media  , 
                                       top30 = ref_cre$`RENDIMENTO DE CARCAÇA MACHOS`$top30  , 
                                       topind = ref_cre$`RENDIMENTO DE CARCAÇA MACHOS`$topind 
                                     )
                     ),
                     lapiNumericInput(inputId = ns('venda_touros_valor'), label="Valor (R$/@)", value = 160, step = 0.01)
      )
    )  
    
    mPanel <- tags$div(
      tagList(
        #tags$h4("DESCRIÇÃO DE ÁREA E LOTAÇÃO", class="lapi-tabletitle"),uiOutput(ns('rebanho')),
        tags$h4("PARÂMETROS GERAIS", class="lapi-tabletitle", style = 'float:left;'), 
        downloadButton(ns("downloadData"),label = "Excel", style = 'float:right; padding-bottom:4px; padding-top:4px;'),
        uiOutput(ns('zooT')),
        tags$h4("Parâmetros de peso do rebanho UA", class="lapi-tabletitle"), 
        uiOutput(ns('estimativas')),
        tags$h4("TABELA DE VENDAS", class="lapi-tabletitle"), 
        uiOutput(ns('calcVendas')),
        tags$h4("RESULTADOS", class="lapi-tabletitle"), 
        uiOutput(ns('resultados'))
      )
    )  
    
    sPanel <- tagList(
      ctrlAreaLotacao,
      #ctrlZooTecnicos,
      ctrlEstimativas,
      ctrlVendasForm,
      lapiCollapsablePanel(id=ns("form_desembolso"), "Parâmetros de desembolso",
                           lapiSliderInput(inputId = ns('desembolso_cab_mes'), 
                                           label="Desembolso, R$/cab/mês", 
                                           min=get_desembolso()$domain[1]%>% round(.,2), max=get_desembolso()$domain[5]%>% round(.,2), 
                                           step=.5, value=get_desembolso()$value%>% round(.,2), 
                                           marks=c(get_desembolso()$reference$value%>% round(.,2)), 
                                           marks_colors = c("blue"),
                                           intervals=c(get_desembolso()$domain%>% round(.,2)),grt_is_better = F,
                                           tooltip = write_ref_tooltip(
                                             fazenda = ref_farm$`DESEMBOLSO CAB/MÊS`, 
                                             media = ref_cre$`DESEMBOLSO CAB/MÊS`$media, 
                                             top30 = ref_cre$`DESEMBOLSO CAB/MÊS`$top30, 
                                             topind = ref_cre$`DESEMBOLSO CAB/MÊS`$topind
                                           )
                           )
      )
    )
    
    fluidPage(
      fluidRow(
        tags$div(class="col-xs-12 col-sm-5 col-md-4 col-lg-4 lapi-sidepanel", sPanel, 'ss-container' = NA),
        tags$div(class="col-xs-12 col-sm-7 col-md-8 col-md-8 lapi-mainpanel", mPanel),
        tags$script({
          "
            $(function(){
              var h = $(window).height();
              var new_h = h * 0.80;
              $('#ciclo-ciclo_completo .lapi-sidepanel').css('height', new_h);
              $('#ciclo-ciclo_completo .lapi-mainpanel').css('height', new_h);
              
              //var el = document.querySelector('#ciclo-ciclo_completo .lapi-sidepanel');
              //SimpleScrollbar.initEl(el);
              
            });
          "
        })
    ))
  })
  
  
  output$downloadData <- downloadHandler(
    
    filename = function() {
      paste0('metas_ciclo_completo_',Sys.Date(), ".xlsx", sep = "")
    },
    
    content = function(file) {
      
      tibble(
        ## PRIMEIRO GRUPO
        "Área produtiva, ha" = input$area_produtiva %>% round,
        "Lotação, UA/ha" = input$lotacao_area %>% round(.,2),
        "Lotação, cab/ha" = input$lotacao_cabeca %>% round(.,2),
        "Participação matrizes, %" = input$participacao_total %>% round(.,2),
        "Fertilidade, %" = input$fertilidade %>% round(.,2),
        "Perda pré-parto, %" = input$perda_pre_parto %>% round(.,2),
        "Mortalidade bezerros, %" = input$mortalidade_bezerros %>% round(.,2),
        "Idade ao desmame, m" = input$idade_desmame %>% round,
        "Venda de bezerros machos, %" = input$venda_bezerros_machos %>% round(.,2),
        "Venda de bezerras fêmeas, %" = input$venda_bezerras_femeas %>% round(.,2),
        "Idade de venda de novilhas não entouradas, m" = input$idade_nao_entouradas %>% round,
        "Idade ao abate, m" = input$idade_abate %>% round,
        ### SEGUNDO GRUPO
        "Matrizes" = input$peso_matrizes  %>% round(.,2),
        "Bezerros mamando" = input$peso_bezerros_mamando  %>% round(.,2),
        "Fêmeas 8 a 12 meses" = input$peso_f_12 %>% round(.,2),
        "Fêmeas 13 a 24 meses" = input$peso_f_24 %>% round(.,2),
        "Machos 8 as 12 meses" = input$peso_m_12 %>% round(.,2),
        "Machos 13 a 24 meses" = input$peso_m_24 %>% round(.,2),
        "Machos +24 meses" = input$peso_m_mais %>% round(.,2),
        "Touros" = input$peso_touros %>% round(.,2),
        ### TERCEIRO GRUPO
        "Vacas descarte, Peso, kg" = input$venda_vc_peso %>% round(.,2),
        "Vacas descarte, Rendimento de carcaça, %" = input$rend_carc_vc %>% round(.,2),
        "Vacas descarte, Valor, R$/@" = input$venda_descarte_valor %>% round(.,2),
        "Desmame machos, Peso, kg" = input$venda_macho_peso %>% round(.,2),
        "Desmame machos, Valor, R$/kg" = input$venda_macho_valor %>% round(.,2),
        "Desmame fêmeas, Peso, kg" = input$venda_femea_peso %>% round(.,2),
        "Desmame fêmeas, Valor, R$/kg" = input$venda_femea_valor %>% round(.,2),
        "Venda de bois, Peso, kg" = input$venda_peso_bois %>% round(.,2),
        "Venda de bois, Rendimento de carcaça, %" = input$rend_carc_bois %>% round(.,2),
        "Venda de bois, Valor, R$/@" = input$venda_bois_valor %>% round(.,2),
        "Novilhas descarte, Peso, kg" = input$venda_peso_novilhas %>% round(.,2),
        "Novilhas descarte, Rendimento de carcaça, %" = input$rend_carc_novilhas %>% round(.,2),
        "Novilhas descarte, Valor, R$/@" = input$venda_novilhas_valor %>% round(.,2),
        "Touros descarte, Peso, kg" = input$peso_touros %>% round(.,2),
        "Touros descarte, Rendimento de carcaça, %" = input$rend_carc_touros %>% round(.,2),
        "Touros descarte, Valor, R$/@" = input$venda_touros_valor %>% round(.,2),
        ### QUARTO GRUPO
        "Desembolso, R$/cab/mês" = input$desembolso_cab_mes %>% round(.,2)
      ) %>% 
        pivot_longer(everything(), names_to = 'Parâmetro', values_to = 'Valor') %>% 
        mutate('Tipo parâmetro' = c(
          rep("Gerais", 12),
          rep("Peso do rebanho, UA", 8),
          rep("Venda/Abate", 16),
          rep("Desembolso", 1)
        )) %>% 
        select(3,1,2) %>% 
        as.data.frame() -> entradas ## ENTRADAS
      
      to_export$zoot %>% 
        rename("Descrição" = 1, "Unidade" = 2, "Valor" = 3) %>% 
        as.data.frame() -> gerais ## GERAIS
        
      to_export$estimativas %>% 
        rename("Categoria" = 1, "Cabeças" = 2) %>% 
        as.data.frame() -> peso_rebanho ## PESO REBANHO
       
      to_export$vendas_calc %>% 
        rownames_to_column() %>% 
        rename("Categoria" = 1, "Quantidade" = 2, "Peso (@)" = 3, "Valor/cab" = 4, "Valor total" = 5) %>% 
        as.data.frame() -> vendas ## VENDAS
       
      to_export$resultados %>% 
        rename("Resultado" = 1, "R$" = 2) %>% 
        as.data.frame() -> resultados ## RESULTADOS
      
      wb <- createWorkbook(creator = "Inttegra")
      
      addWorksheet(wb, "Entradas")
      addWorksheet(wb, "Gerais")
      addWorksheet(wb, "Peso Rebanho UA")
      addWorksheet(wb, "Vendas")
      addWorksheet(wb, "Resultados")
      
      writeData(wb, sheet = "Entradas", x = entradas)
      writeData(wb, sheet = "Gerais", x = gerais)
      writeData(wb, sheet = "Peso Rebanho UA", x = peso_rebanho)
      writeData(wb, sheet = "Vendas", x = vendas)
      writeData(wb, sheet = "Resultados", x = resultados)
      
      saveWorkbook(wb, file = file, overwrite = TRUE)
      
    }
  )
  
  
}
    
## To be copied in the UI
# mod_ciclo_completo_ui("ciclo_completo_ui_1")
    
## To be copied in the server
# callModule(mod_ciclo_completo_server, "ciclo_completo_ui_1")
 
